<?php
/**
 * Module Name: Woocommerce Sensei Integration
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) exit;

if ( function_exists( 'Sensei' ) ) :

class WPAchievements_Module_Sensei {

  /**
   * Init required hooks
   *
   * @static
   * @access  public
   * @return  void
   */
  public static function init() {

    // Register settings hooks
    add_filter( 'wpachievements_get_sections_module', array( __CLASS__, 'register_section' ) );
    add_filter( 'wpachievements_achievements_modules_admin_settings', array( __CLASS__, 'admin_settings'), 10, 3 );

    // Register WPAchievements hooks
    add_action( 'wpachievements_admin_events', array( __CLASS__, 'trigger_dropdown' ) );
    add_filter( 'wpachievements_trigger_description', array( __CLASS__, 'selected_trigger' ) );
    add_filter( 'wpachievements_activity_description', array( __CLASS__, 'activity_description'), 10, 4 );
    add_filter( 'wpachievements_quest_description', array( __CLASS__, 'quest_description' ), 10, 3 );

    // Register Sensei hooks
    add_action( 'sensei_course_status_updated', array( __CLASS__, 'completed_course' ), 10, 3 );
    add_action( 'sensei_user_quiz_grade', array( __CLASS__, 'graded_quiz' ), 10, 4 );
    add_action( 'sensei_lesson_status_updated', array( __CLASS__, 'completed_lesson' ), 10, 3 );
  }

  /**
   * Triggered on course status change
   *
   * @static
   * @access  public
   * @param   string $status    Course status: in-progress, complete ...
   * @param   int $user_id      User ID
   * @param   int $course_id    Course post ID
   * @return  void
   */
  public static function completed_course( $status, $user_id, $course_id ) {

    if ( 'complete' != $status ) {
      return;
    }
    WPAchievements_Trigger::new_activity( array(
      'activity'    => 'sensei_course_complete',
      'user_id'     => $user_id,
      'post_id'     => $course_id,
      'points'      => (int) wpachievements_get_site_option('wpachievements_sensei_course_complete'),
    ) );
  }

  /**
   * Triggered on quiz complete
   *
   * @static
   * @access  public
   * @param   int $user_id      User ID
   * @param   int $quiz_id      Quiz post ID
   * @param   float $grade      Quiz grade
   * @param   float $passmark   Quiz passmark (Percentage)
   * @return  void
   */
  public static function graded_quiz( $user_id, $quiz_id, $grade, $passmark ) {

    if ( $grade >= $passmark ) {
      $activity_type = 'sensei_quiz_pass';

      if ( 100 == $grade ) {
        $activity_type = 'sensei_quiz_perfect';
      }
    }
    else {
      $activity_type = 'sensei_quiz_fail';
    }
    WPAchievements_Trigger::new_activity( array(
      'activity'    => $activity_type,
      'user_id'     => $user_id,
      'post_id'     => $quiz_id,
      'points'      => (int) wpachievements_get_site_option( 'wpachievements_' . $activity_type),
    ) );    
  }

  /**
   * Triggered on lesson status change
   *
   * @static
   * @access  public
   * @param   string $status    Lesson status: in-progress, complete ...
   * @param   int $user_id      User ID
   * @param   int $course_id    Lesson post ID
   * @return  void
   */
  public static function completed_lesson( $status, $user_id, $lesson_id ) {

    if ( 'complete' != $status && 'passed' != $status ) {
      return;
    }
    WPAchievements_Trigger::new_activity( array(
      'activity'    => 'sensei_lesson_complete',
      'user_id'     => $user_id,
      'post_id'     => $lesson_id,
      'points'      => (int) wpachievements_get_site_option( 'wpachievements_sensei_lesson_complete' ),
    ) );       
  }

  /**
   * Register admin section
   *
   * @static
   * @access  public
   * @param   array $sections Array of registered sections for admin settings
   * @return  array           Sections
   */
  public static function register_section( $sections ) {
    $sections['sensei'] = __("Sensei", 'wpachievements' );

    return $sections;
  }

  /**
   * Admin settings
   *
   * @static
   * @access  public
   * @param   array $default_settings
   * @param   string $short_name
   * @param   string $section
   * @return  array
   */
  public static function admin_settings( $default_settings, $short_name, $section ) {

    if ( $section !== 'sensei' ) {
      return $default_settings;
    }

    $settings = array();
    $settings[] = array( 'title' => __( 'Sensei', 'wpachievements' ), 'type' => 'title', 'desc' => '', 'id' => 'Sensei_options' );

    $settings[] = array(
      'title'   => __( 'User Passes Quiz', 'wpachievements' ),
      'desc'    => __( 'Points awarded when a user passes a quiz.', 'wpachievements' ),
      'id'      => $short_name.'_sensei_quiz_pass',
      'type'    => 'number',
      'default' => '0',
    );

    $settings[] = array(
      'title'   => __( 'User Fails Quiz', 'wpachievements' ),
      'desc'    => __( 'Points awarded when a user fails a quiz.', 'wpachievements' ),
      'id'      => $short_name.'_sensei_quiz_fail',
      'type'    => 'number',
      'default' => '0',
    );

    $settings[] = array(
      'title'   => __( 'User Gets Perfect Quiz Score', 'wpachievements' ),
      'desc'    => __( 'Points awarded when a user gets 100% on a quiz.', 'wpachievements' ),
      'id'      => $short_name.'_sensei_quiz_perfect',
      'type'    => 'number',
      'default' => '0',
    );

    $settings[] = array(
      'title'   => __( 'User Completes Lesson', 'wpachievements' ),
      'desc'    => __( 'Points awarded when a user completes a lesson.', 'wpachievements' ),
      'id'      => $short_name.'_sensei_lesson_complete',
      'type'    => 'number',
      'default' => '0',
    );

    $settings[] = array(
      'title'   => __( 'Users Completes Course.', 'wpachievements' ),
      'desc'    => __( 'Points awarded when a user completes a course.', 'wpachievements' ),
      'id'      => $short_name.'_sensei_course_complete',
      'type'    => 'number',
      'default' => '0',
    );

    $settings[] = array( 'type' => 'sectionend', 'id' => 'Sensei_options');

    return $settings;
  }

  /**
   * Return all available triggers with descriptions
   *
   * @static
   * @access  public
   * @return  array Array of triggers and descriptions
   */
  public static function get_triggers() {
    return array(
      "sensei_quiz_pass"        => __('The user passes a quiz', 'wpachievements'),
      "sensei_quiz_fail"        => __('The user fails a quiz', 'wpachievements'),
      "sensei_quiz_perfect"     => __('The user gets 100% on a quiz', 'wpachievements'),
      "sensei_lesson_complete"  => __('The user completes a lesson', 'wpachievements'),
      "sensei_course_complete"  => __('The user completes a course', 'wpachievements'),
    );
  }

  /**
   * Register trigger events displayed on the achievements page
   *
   * @static
   * @access  public
   * @return  string
   */
  public static function trigger_dropdown() {
    ob_start();
    ?>
    <optgroup label="Sensei Events">
      <?php foreach ( self::get_triggers() as $key => $description ) : ?>
        <option value="<?php echo $key; ?>"><?php echo $description; ?></option>
      <?php endforeach; ?>
    </optgroup>
    <?php
    ob_end_flush();
  }

  /**
   * Return description of selected trigger
   *
   * @static
   * @access  public
   * @param   string $trigger Trigger key
   * @return  string          Trigger description
   */
  public static function selected_trigger( $trigger ) {
    $triggers = self::get_triggers();

    if ( array_key_exists( $trigger, $triggers ) ) {
      return $triggers[ $trigger ];
    }

    return $trigger;
  }

  /**
   * Generate activity descriptions
   *
   * @static
   * @access  public
   * @param   string $text  Activity description
   * @param   string $type  Activity trigger
   * @param   int $points   Points
   * @param   int $times    Occurrences
   * @return  string        Activity description
   */
  public static function activity_description( $text, $type, $points, $times ) {

    switch( $type ) {
      case 'sensei_quiz_pass': {
        $text = sprintf( _n( 'for passing %s quiz', 'for passing %s quizzes', $times, 'wpachievements' ), $times );
      } break;

      case 'sensei_quiz_fail': {
        $text = sprintf( _n( 'for failing %s quiz', 'for failing %s quizzes', $times, 'wpachievements' ), $times );
      } break;

      case 'sensei_quiz_perfect': {
        $text = sprintf( _n( 'for getting %s perfect score', 'for getting %s perfect scores', $times, 'wpachievements' ), $times );
      } break;

      case 'sensei_lesson_complete': {
        $text = sprintf( _n( 'for completing %s lesson', 'for completing %s lessons', $times, 'wpachievements' ), $times );
      } break;

      case 'sensei_course_complete': {
        $text = sprintf( _n( 'for completing %s course', 'for completing %s courses', $times, 'wpachievements' ), $times );
      } break;
    }

    return $text;
  }

  /**
   * Generate quest step descriptions
   *
   * @static
   * @access  public
   * @param   string $text  Step description
   * @param   string $type  Trigger type
   * @param   int $times    Occurrences
   * @return  string        Step description
   */
  public static function quest_description( $text, $type, $times ) {

    switch( $type ) {
      case 'sensei_quiz_pass': {
        $text = sprintf( _n( 'Pass %s quiz', 'Pass %s quizzes', $times, 'wpachievements' ), $times );
      } break;

      case 'sensei_quiz_fail': {
        $text = sprintf( _n( 'Fail %s quiz', 'Fail %s quizzes', $times, 'wpachievements' ), $times );
      } break;

      case 'sensei_quiz_perfect': {
        $text = sprintf( _n( 'Get %s perfect score', 'Get %s perfect scores', $times, 'wpachievements' ), $times );
      } break;

      case 'sensei_lesson_complete': {
        $text = sprintf( _n( 'Complete %s lesson', 'Complete %s lessons', $times, 'wpachievements' ), $times );
      } break;

      case 'sensei_course_complete': {
        $text = sprintf( _n( 'Complete %s course', 'Complete %s courses', $times, 'wpachievements' ), $times );
      } break;
    }

    return $text;
  }
}

WPAchievements_Module_Sensei::init();

endif;